package br.com.senior.acesso.sam.customvalidationserver.client;

import java.io.IOException;
import java.util.Map;
import java.util.Optional;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;

import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;

import br.com.senior.acesso.sam.customvalidationserver.client.model.PlatformLogin;
import br.com.senior.acesso.sam.customvalidationserver.client.model.PlatformLoginResponse;
import br.com.senior.acesso.sam.customvalidationserver.client.model.Response;
import br.com.senior.acesso.sam.customvalidationserver.config.ConfigProperties;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public abstract class AbstractClient {

    private static Optional<String> token = Optional.empty();
    private static final String ACCESS_TOKEN_KEY = "access_token";
    private static final int RECONNECTION_ATTEMPS = 3;

    @Autowired
    private PlatformClient platformClient;

    @Autowired
    private ConfigProperties configProperties;

    private void login() throws AuthenticationException {        
        int attemps = 0;
        while (!isLogged() && attemps < RECONNECTION_ATTEMPS) {
            log.info("Realizando login na plataforma");
            try {
                ResponseEntity<PlatformLoginResponse> response = platformClient.login(new PlatformLogin(configProperties.getUser(), configProperties.getPass()));
                token = getPlatformToken(response.getBody().getJsonToken());
            } catch (Exception e) {
                log.error("Não foi possível realizar o login", e);
                attemps++;
            }
        }
        if (!isLogged()) {
            throw new AuthenticationException("Não foi possível realizar o login na plataforma");
        }
    }

    protected String getToken() throws AuthenticationException {
        login();
        return "Bearer " + token.get();
    }

    private boolean isLogged() {
        return token.isPresent();
    }

    protected Response executeService(Callback callback) throws AuthenticationException {        
        ResponseEntity<?> response = callback.execute();
        if (response.getStatusCode() == HttpStatus.UNAUTHORIZED) {
            login();
            response = callback.execute();
        }
        return (Response) response.getBody();
    }

    private Optional<String> getPlatformToken(String jwtToken) throws JsonParseException, JsonMappingException, IOException {
        ObjectMapper mapper = new ObjectMapper();
        Map<String, Object> map = mapper.readValue(jwtToken, new TypeReference<Map<String, Object>>() {});
        return Optional.of((String) map.get(ACCESS_TOKEN_KEY));
    }
}

interface Callback {
    public ResponseEntity<?> execute() throws AuthenticationException;
}
